<?php

namespace app\controllers;

use app\models\Detallefacturas;
use app\models\Parametros;
use app\models\Productos;
use app\models\ProductosSearch;
use app\models\User;
use Exception;
use SebastianBergmann\CodeCoverage\Report\Xml\Totals;
use stdClass;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;
use Yii;
use yii\web\UploadedFile;

/**
 * ProductosController implements the CRUD actions for Productos model.
 */
class ProductosController extends Controller
{
    /**
     * @inheritDoc
     */
    public function behaviors()
    {
        return array_merge(
            parent::behaviors(),
            [
                'verbs' => [
                    'class' => VerbFilter::className(),
                    'actions' => [
                        'delete' => ['POST'],
                    ],
                ],
            ]
        );
    }


    public function beforeAction($action)
    {


        if (in_array($action->id, ['ver-producto', 'ver-catalogo'])) {
            return parent::beforeAction($action);
        }


        $userModel = Yii::$app->user->identity;

        // Redirige si el usuario no está logueado
        if (Yii::$app->user->isGuest) {
            return $this->redirect(['site/login']);
        }

        // Redirige si el usuario no es superAdmin para las acciones update, create, delete
        if (in_array($action->id, ['update', 'create', 'delete'])) {
            if ($userModel->rol !== 'superAdmin') {
                return $this->redirect(['site/access-denied']);
            }
        }

        // Redirige si el usuario no es superAdmin ni vendedor para la acción index
        if ($action->id === 'index') {
            if (!in_array($userModel->rol, ['superAdmin', 'vendedor'])) {
                return $this->redirect(['site/access-denied']);
            }
        }

        return parent::beforeAction($action);
    }

    /**
     * Lists all Productos models.
     *
     * @return string
     */
    public function actionIndex()
    {
        $this->layout = 'dashboard';
        date_default_timezone_set('America/Guayaquil'); //1:::HORA DE GUAYAQUIL
        $searchModel = new ProductosSearch();
        $dataProvider = $searchModel->search($this->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single Productos model.
     * @param int $pro_id Pro ID
     * @return string
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionView($pro_id)
    {
        $this->layout = 'dashboard';
        return $this->render('view', [
            'model' => $this->findModel($pro_id),
        ]);
    }


    public function actionVerProducto($id)
    {

        $this->layout = 'empty';
        return $this->render('verproducto', [
            'model' => $this->findModel($id),
        ]);
    }


    public function actionVerCatalogo()
    {

        $this->layout = 'empty';

        $catalogo = Productos::find()->where(['pro_estado' => Parametros::ACTIVO])->all();
        return $this->render('vercatalogo', [
            'models' => $catalogo,
        ]);
    }

    /**
     * Creates a new Productos model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return string|\yii\web\Response
     */
    public function actionCreate()
    {
        date_default_timezone_set('America/Guayaquil'); //1:::HORA DE GUAYAQUIL
        $model = new Productos();
        $this->layout = 'dashboard';

        if ($model->load(Yii::$app->request->post())) {


            $model->created_by = Yii::$app->user->id;
            $model->updated_by = Yii::$app->user->id;
            $model->created_at = date('Y-m-d H:i:s');
            $model->updated_at = date('Y-m-d H:i:s');



            $user = User::find()->where(['id' => Yii::$app->user->id])->one();


            $fecha = date('Y-m-d H:i:s');
            $detalle = "Creación de producto desde el backOffice";

            $detalleStock = "<i class='bx bxs-plus-circle bx-xs'></i>  {$fecha}    -   Cant: {$model->pro_stock}   -  Det: {$detalle} -  Resp: $user->nombre";

            $model->pro_aumentar_stock = $detalleStock  . "|";


            $imagen = UploadedFile::getInstance($model, 'pro_foto');


            if (!is_null($imagen)) {
                $namePartsImagen = explode(".", $imagen->name);

                $ext = end($namePartsImagen);
                $model->pro_foto = Yii::$app->security->generateRandomString() . ".{$ext}";
                Yii::$app->params['uploadPath'] = Yii::$app->basePath . '/web/uploads/images/';
                $path = Yii::$app->params['uploadPath'] . $model->pro_foto;
                $imagen->saveAs($path);
            }




            if ($model->save()) {
                return $this->redirect(['index']);
            } else {
                $model->loadDefaultValues();
            }
        }


        return $this->render('create', [
            'model' => $model,
        ]);
    }




    /**
     * Updates an existing Productos model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param int $pro_id Pro ID
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionUpdate($pro_id)
    {
        date_default_timezone_set('America/Guayaquil'); //1:::HORA DE GUAYAQUIL
        $this->layout = 'dashboard';
        $model = $this->findModel($pro_id);

        // Guardar el valor anterior de la imagen antes de cargar los datos del formulario
        $oldFoto = $model->pro_foto;

        $model->updated_by = Yii::$app->user->id;
        $model->updated_at = date('Y-m-d H:i:s');

        if ($model->load(Yii::$app->request->post())) {

            // Obtener la nueva imagen
            $imagen = UploadedFile::getInstance($model, 'pro_foto');

            if (!is_null($imagen)) {
                // Si hay una nueva imagen, procesarla
                $namePartsImagen = explode(".", $imagen->name);
                $ext = end($namePartsImagen);
                $model->pro_foto = Yii::$app->security->generateRandomString() . ".{$ext}";
                Yii::$app->params['uploadPath'] = Yii::$app->basePath . '/web/uploads/images/';
                $path = Yii::$app->params['uploadPath'] . $model->pro_foto;
                $imagen->saveAs($path);
            } else {
                // Si no se sube una nueva imagen, mantener la imagen anterior
                $model->pro_foto = $oldFoto;
            }

            // Guardar el modelo
            if ($model->save()) {
                return $this->redirect(['view', 'pro_id' => $model->pro_id]);
            } else {
                $model->loadDefaultValues();
                var_dump($model->getErrors());
                die();
            }
        }

        return $this->render('update', [
            'model' => $model,
        ]);
    }

    /**
     * Deletes an existing Productos model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param int $pro_id Pro ID
     * @return \yii\web\Response
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function actionDelete($pro_id)
    {


        $model =  $this->findModel($pro_id);
        $model->pro_estado = Parametros::ELIMINADO;
        $model->save();

        return $this->redirect(['index']);
    }

    /**
     * Finds the Productos model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param int $pro_id Pro ID
     * @return Productos the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($pro_id)
    {
        if (($model = Productos::findOne(['pro_id' => $pro_id])) !== null) {
            return $model;
        }

        throw new NotFoundHttpException('The requested page does not exist.');
    }



    public function actionGetProducto($codigo)
    {
        $resultado = new \stdClass();
        $resultado->resultado = false;


        $producto = Productos::find()->where(['pro_codigo' => $codigo])->one();

        if (!empty($producto)) {
            $resultado->data = $producto->toArray();
            $resultado->resultado = true;
        } else {
            $resultado->mensaje = 'Producto no encontrado';
        }

        return json_encode($resultado);
    }




    public function actionAddProductoFactura($codigo, $nomProducto, $cantidad, $factura, $metodoPago)
    {
        $resultado = new \stdClass;
        $resultado->transaccion = false;

        try {

            // Buscar el producto por código
            $producto = Productos::find()->where(['pro_codigo' => $codigo, 'pro_estado' => Parametros::ACTIVO])->one();

            if (!$producto) {
                $resultado->error = 'El producto no existe en la base de datos';
                return json_encode($resultado);
            }



            // Validar stock
            if ($producto->pro_stock < $cantidad) {
                $resultado->error = "La cantidad ingresada supera al stock disponible ( $producto->pro_stock items disp)";
                return json_encode($resultado);
            }


            //ingresamos el detalle al registro
            $agregaDetalle = self::agregaDetalleFact($factura, $producto->pro_id, $producto->pro_pvp, $producto->pro_pvp_tarjeta, $cantidad, $nomProducto, $producto->pro_codigo, $metodoPago);


            if ($agregaDetalle->transaccion) {
                //debemos disminuir el stock
                $actualizaStock =  self::desminuirPedidoStock($codigo, $cantidad);

                if ($actualizaStock->transaccion) {
                    $resultado->transaccion = true;
                } else {
                    return json_encode($actualizaStock);
                }
            } else {
                $resultado->error = $agregaDetalle;
            }
        } catch (\Exception $e) {
            $resultado->error = 'Error al agregar el producto: ' . $e->getMessage();
            $resultado->linea = $e->getLine();
        }

        return json_encode($resultado);
    }




    public static function agregaDetalleFact($factura, $pro_id, $pro_pvp, $pro_pvp_tarjeta, $cantidad, $nomProducto, $codigoProd, $metodoPago)
    {

        date_default_timezone_set('America/Guayaquil'); //1:::HORA DE GUAYAQUIL
        $resultado = new \stdClass;
        $resultado->transaccion = false;


        try {

            // Buscar el producto por código
            $producto = Productos::find()->where(['pro_id' => $pro_id, 'pro_estado' => Parametros::ACTIVO])->select(['pro_nombre', 'pro_codigo',  'pro_porc_dscto', 'pro_cant_min_desct',  'pro_costo', 'pro_pvp', 'pro_dscto', 'pro_pvp_tarjeta', 'pro_stock'])->one();

            //Validamos si ya existe un producto con ese detalle
            $detalleFact = Detallefacturas::find()->where(['fac_id' => $factura, 'pro_id' => $pro_id, 'det_estado' => Parametros::ACTIVO])->one();

            if (!empty($detalleFact)) {
                $cantidad = $detalleFact->det_cantidad + $cantidad;
            } else {
                $detalleFact = new Detallefacturas();
                $detalleFact->created_at = date('Y-m-d H:i:s');
            }

            $subtotalProducto = self::calculaValor($producto, $cantidad, $metodoPago);

            $detalleFact->fac_id = $factura;
            $detalleFact->pro_id = $pro_id;
            $detalleFact->pro_codigo = $codigoProd;
            $detalleFact->det_cantidad = $cantidad;
            $detalleFact->det_val_unitario = $pro_pvp;
            $detalleFact->det_val_unitario_tarjeta = $pro_pvp_tarjeta;
            $detalleFact->det_descuento = $subtotalProducto->descuento;
            $detalleFact->det_subtotal = $subtotalProducto->totalProd;
            $detalleFact->det_detalle = $nomProducto;

            $detalleFact->id_vendedor = '';

            $detalleFact->created_by = Yii::$app->user->id;
            $detalleFact->updated_by = Yii::$app->user->id;
            $detalleFact->created_at = date('Y-m-d H:i:s');
            $detalleFact->updated_at = date('Y-m-d H:i:s');

            $detalleFact->det_descuento_tarjeta = $subtotalProducto->descuentoTarjeta;
            $detalleFact->det_subtotal_tarjeta = $subtotalProducto->totalProdTarjeta;

            $detalleFact->det_estado = Parametros::ACTIVO;

            $detalleFact->updated_at = date('Y-m-d H:i:s');

            if ($detalleFact->save()) {
                $resultado->transaccion = true;
            } else {
                $resultado->error = $detalleFact->getErrors();
            }
        } catch (\Exception $e) {
            $resultado->error = 'Error al agregar al detalle: ' . $e->getMessage();
        }


        return $resultado;
    }



    public static function calculaValor($producto, $cantidad, $metodoPago)
    {

        $resultado = new \stdClass;
        $resultado->transaccion = false;
        try {
            // Inicializar variables
            $subtotal = $cantidad * $producto->pro_pvp;
            $subtotalTarjeta = $cantidad * $producto->pro_pvp_tarjeta;


            $desc = 0;
            $descTarjeta = 0;
            $porc_desc = 0;
            $totalProd = 0;
            $totalProdTarjeta = 0;

            // Validar si tiene descuento
            if ($producto->pro_dscto && ($cantidad >= $producto->pro_cant_min_desct)) {


                $desc = $subtotal * ($producto->pro_porc_dscto / 100);
                $porc_desc = $producto->pro_porc_dscto ?? 0;
                $totalProd = $subtotal - $desc;


                //    $descTarjeta = $subtotalTarjeta * ($producto->pro_porc_dscto / 100);
                $totalProdTarjeta = $subtotalTarjeta - $descTarjeta;
            } else {
                $totalProd  = $subtotal;
                $totalProdTarjeta  = $subtotalTarjeta;
            }

            // Asignar los resultados
            $resultado->subtotal = $subtotal;
            $resultado->descuento = $desc;
            $resultado->totalProd = $totalProd;


            $resultado->subtotalTarjeta = $subtotalTarjeta;
            $resultado->descuentoTarjeta = $descTarjeta;
            $resultado->totalProdTarjeta = $totalProdTarjeta;


            $resultado->porcDescuento = $porc_desc;
        } catch (\Exception $e) {
            $resultado->error = 'Error al calcular valor: ' . $e->getMessage();
        }


        return $resultado;
    }



    public static function desminuirPedidoStock($codigo, $cantidad)
    {

        $resultado = new \stdClass;
        $resultado->transaccion = false;


        try {

            $producto = Productos::find()->where(['pro_codigo' => $codigo])->one();

            if (!$producto) {
                $resultado->error = 'El producto no existe en la base de datos';
                return $resultado;
            }


            if ($producto->pro_stock < $cantidad) {
                $resultado->error = "La cantidad a dar de baja es mayor al stock disponible {$producto->pro_stock}";
                return $resultado;
            }

            $nuevaCantidad = $producto->pro_stock - $cantidad;
            $producto->pro_stock = $nuevaCantidad;

            if ($producto->save(false, ['pro_stock'])) {
                $resultado->transaccion = true;
            } else {
                $resultado->error = $producto->getErrors();
            }
        } catch (\Exception $e) {
            $resultado->error = 'Error al actualizar stock: ' . $e->getMessage();
        }


        return $resultado;
    }


    public static function aumentarProductoStock($codigo, $cantidad)
    {

        $resultado = new \stdClass;
        $resultado->transaccion = false;


        try {

            $producto = Productos::find()->where(['pro_id' => $codigo])->one();

            if (!$producto) {
                $resultado->error = 'El producto no existe en la base de datos' . $codigo;
                return json_encode($resultado);
            }


            $nuevaCantidad = $producto->pro_stock + $cantidad;
            $producto->pro_stock = $nuevaCantidad;

            if ($producto->save(false, ['pro_stock'])) {
                $resultado->transaccion = true;
            } else {
                $resultado->error = $producto->getErrors();
            }
        } catch (\Exception $e) {
            $resultado->error = 'Error al aumentar stock: ' . $e->getMessage();
        }


        return $resultado;
    }


    public static function actionEliminarDetalle($idDetalle, $factura)
    {

        $resultado = new \stdClass;
        $resultado->transaccion = false;


        try {

            //$producto = Productos::find()->where(['pro_codigo' => $codigo])->one();

            $registro = Detallefacturas::find()->where(['fac_id' => $factura, 'det_id' => $idDetalle])->one();
            $cantidadAum = $registro->det_cantidad;
            $prod = $registro->pro_id;


            if ($registro->delete()) {
                $resultado = self::aumentarProductoStock($prod, $cantidadAum);
                // $resultado->transaccion = true;
            } else {
                $resultado->error = $registro->getErrors();
            }
        } catch (\Exception $e) {
            $resultado->error = 'Error al eliminar detalle: ' . $e->getMessage();
        }


        return json_encode($resultado);
    }


    public static function actionRestarDetalle($idDetalle, $factura, $metodoPago)
    {

        $resultado = new \stdClass;
        $resultado->transaccion = false;


        try {
            //$producto = Productos::find()->where(['pro_codigo' => $codigo])->one();
            $detalleFact = Detallefacturas::find()->where(['fac_id' => $factura, 'det_id' => $idDetalle])->one();
            $cantidadrest = 1;

            $producto = Productos::find()->where(['pro_id' => $detalleFact->pro_id])->one();

            $detalleFact->det_cantidad = $detalleFact->det_cantidad - $cantidadrest;

            if ($detalleFact->det_cantidad < 1) {
                $resultado->error = '';
                return json_encode($resultado);
            }



            $subtotalProducto = self::calculaValor($producto, $detalleFact->det_cantidad, $metodoPago);

            $detalleFact->det_descuento = $subtotalProducto->descuento;
            $detalleFact->det_subtotal = $subtotalProducto->totalProd;

            $detalleFact->updated_by = Yii::$app->user->id;
            $detalleFact->updated_at = date('Y-m-d H:i:s');

            $detalleFact->det_descuento_tarjeta = $subtotalProducto->descuentoTarjeta;
            $detalleFact->det_subtotal_tarjeta = $subtotalProducto->totalProdTarjeta;


            $detalleFact->updated_at = date('Y-m-d H:i:s');



            if ($detalleFact->save()) {
                $resultado = self::aumentarProductoStock($detalleFact->pro_id, $cantidadrest);
                // $resultado->transaccion = true;
            } else {
                $resultado->error = $detalleFact->getErrors();
            }
        } catch (\Exception $e) {
            $resultado->error = 'Error al restar cantidad: ' . $e->getMessage();
        }

        return json_encode($resultado);
    }







    public function actionSumarDetalle($idDetalle, $factura, $metodoPago)
    {
        $resultado = new \stdClass;
        $resultado->transaccion = false;

        try {
            $detalleFact = Detallefacturas::find()->where(['fac_id' => $factura, 'det_id' => $idDetalle])->one();
            $producto = Productos::find()->where(['pro_id' => $detalleFact->pro_id])->one();

            if (!$producto) {
                $resultado->error = 'El producto no existe en la base de datos';
                return json_encode($resultado);
            }


            $cantidadaum = 1;


            // Validar stock
            if ($producto->pro_stock < $cantidadaum) {
                $resultado->error = "La cantidad ingresada supera al stock disponible ( $producto->pro_stock items disp)";
                return json_encode($resultado);
            }


            //ingresamos el detalle al registro
            $agregaDetalle = self::agregaDetalleFact($factura, $producto->pro_id, $producto->pro_pvp, $producto->pro_pvp_tarjeta, $cantidadaum, $detalleFact->det_detalle, $producto->pro_codigo, $metodoPago);



            if ($agregaDetalle->transaccion) {
                //debemos disminuir el stock
                $actualizaStock =  self::desminuirPedidoStock($producto->pro_codigo, $cantidadaum);

                if ($actualizaStock->transaccion) {
                    $resultado->transaccion = true;
                } else {
                    return json_encode($actualizaStock);
                }
            } else {
                $resultado->error = $agregaDetalle;
            }
        } catch (\Exception $e) {
            $resultado->error = 'Error al agregar el producto: ' . $e->getMessage();
            $resultado->linea = $e->getLine();
        }

        return json_encode($resultado);
    }



    public static function actionAgregarStock($id, $numFactura, $detalle, $cantidad)
    {
        $resultado = new \stdClass;
        $resultado->transaccion = false;
        date_default_timezone_set('America/Guayaquil'); //1:::HORA DE GUAYAQUIL

        try {

            $user = User::find()->where(['id' => Yii::$app->user->id])->one();
            $producto = Productos::find()->where(['pro_id' => $id])->one();

            $fecha = date('Y-m-d H:i:s');
            $detalleAct = !empty($producto->pro_aumentar_stock) ? $producto->pro_aumentar_stock : '';
            $detalleStock = "<i class='bx bxs-plus-circle bx-xs'></i>  {$fecha}    -   Num. fact: {$numFactura}   -  Cant: {$cantidad}   -  Det: {$detalle} -  Resp: $user->nombre";

            $producto->pro_aumentar_stock = $detalleAct . $detalleStock . "|";
            $producto->updated_at = date('Y-m-d H:i:s');
            $producto->pro_stock =  $producto->pro_stock + $cantidad;

            $resultado->transaccion = true;


            if ($producto->save(false, ['updated_at', 'pro_aumentar_stock', 'pro_stock'])) {
                $resultado->transaccion = true;
            } else {
                $resultado->error = $producto->getErrors();
            }
        } catch (\Exception $e) {
            // Manejo de errores
            $resultado->error = 'Error al obtener detalle: ' . $e->getMessage();
        }

        // Retornar el resultado en formato JSON
        return json_encode($resultado);
    }



    public static function actionDisminuirStock($id, $detalle, $cantidad)
    {
        $resultado = new \stdClass;
        $resultado->transaccion = false;
        date_default_timezone_set('America/Guayaquil'); //1:::HORA DE GUAYAQUIL

        try {

            $user = User::find()->where(['id' => Yii::$app->user->id])->one();
            $producto = Productos::find()->where(['pro_id' => $id])->one();

            if ($cantidad > $producto->pro_stock) {
                $resultado->error = 'La cantidad excede al stock actual';
                return json_encode($resultado);
            }

            $fecha = date('Y-m-d H:i:s');
            $detalleAct = !empty($producto->pro_aumentar_stock) ? $producto->pro_aumentar_stock : '';
            $detalleStock = "<i class='bx bxs-minus-circle bx-xs'></i>  {$fecha}    -   Cant: {$cantidad}   -  Det: {$detalle} -  Resp: $user->nombre";

            $producto->pro_aumentar_stock = $detalleAct . $detalleStock . "|";
            $producto->updated_at = date('Y-m-d H:i:s');
            $producto->pro_stock =  $producto->pro_stock - $cantidad;

            $resultado->transaccion = true;


            if ($producto->save(false, ['updated_at', 'pro_aumentar_stock', 'pro_stock'])) {
                $resultado->transaccion = true;
            } else {
                $resultado->error = $producto->getErrors();
            }
        } catch (\Exception $e) {
            // Manejo de errores
            $resultado->error = 'Error al obtener detalle: ' . $e->getMessage();
        }

        // Retornar el resultado en formato JSON
        return json_encode($resultado);
    }
}
