<?php

namespace app\models;

use Yii;

/**
 * This is the model class for table "clientes".
 *
 * @property int $cli_id
 * @property string $cli_identificacion
 * @property string $cli_nombre
 * @property string $cli_direccion
 * @property string $cli_correo
 * @property string $cli_telefono
 * @property string $cli_estado
 * @property string $cli_celular
 * @property string $cli_tipo_identificacion
 * @property string $created_at
 * @property string $updated_at
 * @property int $created_by
 * @property int $updated_by
 */
class Clientes extends \yii\db\ActiveRecord
{
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'clientes';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['cli_identificacion', 'cli_nombre',  'cli_estado', 'cli_tipo_identificacion', 'created_at', 'updated_at', 'created_by', 'updated_by'], 'required'],
            [['created_at', 'updated_at'], 'safe'],
            [['created_by', 'updated_by'], 'integer'],
            [['cli_identificacion'], 'string', 'max' => 13],
            [['cli_identificacion'], 'unique'],
            [['cli_identificacion'], 'match', 'pattern' => '/^\d+$/', 'message' => 'La identificación solo puede contener dígitos.'],
            [['cli_identificacion'], 'validateIdentificacion'],
            [['cli_nombre'], 'match', 'pattern' => '/^[\p{L}\s]+$/u', 'message' => 'El nombre solo puede contener letras'],
            [['cli_direccion'], 'string', 'max' => 200],
            [['cli_correo'], 'email'],
            [['cli_telefono'], 'string', 'max' => 15],
            [['cli_telefono'], 'match', 'pattern' => '/^\d+$/', 'message' => 'El teléfono solo puede contener dígitos.'],
            [['cli_estado', 'cli_tipo_identificacion'], 'string', 'max' => 1],
            [['cli_celular'], 'match', 'pattern' => '/^\d+$/', 'message' => 'El celular solo puede contener dígitos.'],
            [['cli_celular'], 'string', 'max' => 10],
        ];
    }



    public function validateIdentificacion($attribute, $params)
    {
        if ($this->cli_tipo_identificacion == Parametros::CEDULA) {
            // Validar cédula
            $this->validateCedula($attribute, $params);
        } elseif ($this->cli_tipo_identificacion == Parametros::RUC) {
            // Validar RUC
            $this->validateRuc($attribute, $params);
        }
    }

    public function validateCedula($attribute, $params)
    {
        $cedula = $this->$attribute;

        if (!$this->isValidCedula($cedula)) {
            $this->addError($attribute, 'La cédula ingresada no es válida.');
        }
    }

    public function validateRuc($attribute, $params)
    {
        $ruc = $this->$attribute;

        // Verifica que el número tenga 13 dígitos
        if (strlen($ruc) != 13) {
            $this->addError($attribute, 'El RUC debe tener 13 dígitos.');
            return;
        }

        // Los primeros 10 dígitos deben ser una cédula válida
        $cedula = substr($ruc, 0, 10);
        if (!$this->isValidCedula($cedula)) {
            $this->addError($attribute, 'Los primeros diez dígitos del RUC no corresponden a una cédula válida.');
            return;
        }

        // Los últimos 3 dígitos deben estar en el rango de 001 a 999
        $sufijo = substr($ruc, 10, 3);
        if ($sufijo < '001' || $sufijo > '999') {
            $this->addError($attribute, 'Los últimos tres dígitos del RUC no son válidos.');
        }
    }

    private function isValidCedula($cedula)
    {
        // Verifica que el número tenga 10 dígitos
        if (strlen($cedula) != 10) {
            return false;
        }

        // Verifica que los dos primeros dígitos correspondan a una provincia válida (01-24)
        $provincia = substr($cedula, 0, 2);
        if ($provincia < 1 || $provincia > 24) {
            return false;
        }

        // Verifica que el tercer dígito sea menor que 6 (cédula de persona natural)
        $tercerDigito = $cedula[2];
        if ($tercerDigito >= 6) {
            return false;
        }

        // Algoritmo de validación de cédula
        $coeficientes = [2, 1, 2, 1, 2, 1, 2, 1, 2];
        $suma = 0;

        for ($i = 0; $i < 9; $i++) {
            $valor = $cedula[$i] * $coeficientes[$i];
            if ($valor >= 10) {
                $valor -= 9;
            }
            $suma += $valor;
        }

        $digitoVerificador = (10 - ($suma % 10)) % 10;

        return $cedula[9] == $digitoVerificador;
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'cli_id' => 'Cli ID',
            'cli_identificacion' => 'Identificación',
            'cli_nombre' => 'Nombre',
            'cli_direccion' => 'Dirección',
            'cli_correo' => 'Correo electrónico',
            'cli_telefono' => 'Teléfono',
            'cli_estado' => 'Estado',
            'cli_celular' => 'Celular',
            'cli_tipo_identificacion' => 'Tipo Identificación',
            'created_at' => 'Fecha creación',
            'updated_at' => 'Fecha actualización',
            'created_by' => 'Creado por',
            'updated_by' => 'Actualizado por',
        ];
    }
}
